use crate::codegen::rust::nodes::helpers::{node_field_name, struct_name};

fn contents(node: &lib_ruby_parser_nodes::Node) -> String {
    format!(
        "// This file is autogenerated by {generator}

{imports}

#[repr(C)]
pub(crate) struct Internal{struct_name} {{
    {fields_declaration}
}}
",
        generator = file!(),
        imports = imports(&node).join("\n"),
        struct_name = struct_name(node),
        fields_declaration = node.fields.map(field_declaration).join("\n    "),
    )
}

pub(crate) fn codegen(node: &lib_ruby_parser_nodes::Node) {
    let dir = super::filename(node);
    let path = format!("src/nodes/types/{}/internal.rs", dir);
    std::fs::write(&path, contents(node)).unwrap();
}

fn imports(node: &lib_ruby_parser_nodes::Node) -> Vec<&str> {
    use lib_ruby_parser_nodes::NodeFieldType;

    let mut imports = vec![];
    imports.push("use crate::Loc;");

    let has_field = |field_type: NodeFieldType| node.fields.any_field_has_type(field_type);

    if has_field(NodeFieldType::Node)
        || has_field(NodeFieldType::Nodes)
        || has_field(NodeFieldType::MaybeNode {
            regexp_options: true,
        })
        || has_field(NodeFieldType::MaybeNode {
            regexp_options: false,
        })
    {
        imports.push("use crate::Node;");
    }

    if has_field(NodeFieldType::StringValue) {
        imports.push("use crate::Bytes;");
    }

    if has_field(NodeFieldType::MaybeNode {
        regexp_options: true,
    }) || has_field(NodeFieldType::MaybeNode {
        regexp_options: false,
    }) || has_field(NodeFieldType::MaybeLoc)
        || has_field(NodeFieldType::MaybeStr { chars: true })
        || has_field(NodeFieldType::MaybeStr { chars: false })
    {
        imports.push("crate::use_native_or_external!(Maybe);");
    }

    if has_field(NodeFieldType::Node)
        || has_field(NodeFieldType::MaybeNode {
            regexp_options: true,
        })
        || has_field(NodeFieldType::MaybeNode {
            regexp_options: false,
        })
    {
        imports.push("crate::use_native_or_external!(Ptr);");
    }

    if has_field(NodeFieldType::Nodes) {
        imports.push("crate::use_native_or_external!(List);");
    }

    if has_field(NodeFieldType::Str { raw: true })
        || has_field(NodeFieldType::Str { raw: false })
        || has_field(NodeFieldType::MaybeStr { chars: true })
        || has_field(NodeFieldType::MaybeStr { chars: false })
    {
        imports.push("crate::use_native_or_external!(StringPtr);");
    }

    imports
}

fn field_type(field: &lib_ruby_parser_nodes::NodeField) -> &str {
    use lib_ruby_parser_nodes::NodeFieldType;

    match field.field_type {
        NodeFieldType::Node => "Ptr<Node>",
        NodeFieldType::Nodes => "List<Node>",
        NodeFieldType::MaybeNode { .. } => "Maybe<Ptr<Node>>",
        NodeFieldType::Loc => "Loc",
        NodeFieldType::MaybeLoc => "Maybe<Loc>",
        NodeFieldType::Str { .. } => "StringPtr",
        NodeFieldType::MaybeStr { .. } => "Maybe<StringPtr>",
        NodeFieldType::StringValue => "Bytes",
        NodeFieldType::U8 => "u8",
    }
}

fn field_declaration(field: &lib_ruby_parser_nodes::NodeField) -> String {
    format!(
        "#[allow(dead_code)]
    pub(crate) {field_name}: {field_type},",
        field_name = node_field_name(field),
        field_type = field_type(field)
    )
}
