use crate::codegen::cpp::helpers as cpp_helpers;
use lib_ruby_parser_bindings::{
    helpers::messages::{
        constructor::sig as constructor_sig, field_getter::sig as field_getter_sig,
        fields::field_name, variant_getter::sig as variant_getter_sig,
        variant_predicate::sig as variant_predicate_sig,
    },
    Options,
};

fn contents(options: &Options) -> String {
    let messages = lib_ruby_parser_nodes::messages();

    format!(
        "// This file is autogenerated by {generator}

#include \"bindings.hpp\"

extern \"C\"
{{
    {constructors}

    {variant_getters}

    {field_getters}

    {variant_predicates}

    void lib_ruby_parser__external__diagnostic_message__drop(DiagnosticMessage_BLOB *self_blob)
    {{
        DiagnosticMessage *self = (DiagnosticMessage *)self_blob;
        drop_diagnostic_message(self);
    }}
}}
",
        generator = file!(),
        constructors = messages
            .map(|message| constructor(message, options))
            .join("\n\n    "),
        variant_getters = messages
            .map(|message| variant_getter(message, options))
            .join("\n\n    "),
        field_getters = messages
            .flat_map(|message| field_getters(message, options))
            .join("\n\n    "),
        variant_predicates = messages
            .map(|message| variant_predicate(message, options))
            .join("\n\n    ")
    )
}

pub(crate) fn codegen(options: &Options) {
    std::fs::write("external/cpp/bindings_messages.cpp", contents(options)).unwrap();
}

fn constructor(message: &lib_ruby_parser_nodes::Message, options: &Options) -> String {
    let arglist = message
        .fields
        .map(|field| {
            format!(
                "UNPACK_{t}({name}_blob)",
                name = field_name(field),
                t = cpp_helpers::messages::field_type(field)
            )
        })
        .join(", ");

    format!(
        "{sig}
    {{
        return PACK_DiagnosticMessage(DiagnosticMessage({inner_t}({arglist})));
    }}",
        sig = constructor_sig(message, options),
        inner_t = message.camelcase_name,
        arglist = arglist
    )
}
fn variant_getter(message: &lib_ruby_parser_nodes::Message, options: &Options) -> String {
    format!(
        "{sig}
    {{
        const DiagnosticMessage *self = (const DiagnosticMessage *)self_blob;
        const {variant_name} *variant = std::get_if<{variant_name}>(&(self->variant));
        return (const {variant_name}_BLOB*)variant;
    }}",
        sig = variant_getter_sig(message, options),
        variant_name = message.camelcase_name,
    )
}
fn field_getters(message: &lib_ruby_parser_nodes::Message, options: &Options) -> Vec<String> {
    message.fields.map(|field| {
        format!(
            "{signature}
    {{
        const {variant_name} *self = (const {variant_name} *)self_blob;
        return (const {blob_type} *)(&(self->{field_name}));
    }}",
            signature = field_getter_sig(message, field, options),
            variant_name = message.camelcase_name,
            field_name = field_name(field),
            blob_type = cpp_helpers::messages::blob_type(field),
        )
    })
}
fn variant_predicate(message: &lib_ruby_parser_nodes::Message, options: &Options) -> String {
    format!(
        "{signature}
    {{
        const DiagnosticMessage *self = (const DiagnosticMessage *)self_blob;
        return std::holds_alternative<{variant_name}>(self->variant);
    }}",
        signature = variant_predicate_sig(message, options),
        variant_name = message.camelcase_name
    )
}
