use crate::codegen::c::helpers::messages::fields::blob_type;
use lib_ruby_parser_bindings::{
    helpers::messages::{
        constructor::sig as constructor_sig, field_getter::sig as field_getter_sig,
        fields::field_name, variant_getter::sig as variant_getter_sig,
        variant_predicate::sig as variant_predicate_sig,
    },
    Options,
};
use lib_ruby_parser_nodes::Message;

fn contents(options: &Options) -> String {
    let messages = lib_ruby_parser_nodes::messages();

    format!(
        "// This file is autogenerated by {generator}
#include \"bindings.h\"

{constructors}

{variant_getters}

{field_getters}

{variant_predicates}

void lib_ruby_parser__external__diagnostic_message__drop(LIB_RUBY_PARSER_DiagnosticMessage_BLOB *self_blob)
{{
    LIB_RUBY_PARSER_DiagnosticMessage *self = (LIB_RUBY_PARSER_DiagnosticMessage *)self_blob;
    LIB_RUBY_PARSER_drop_diagnostic_message(self);
}}
",
        generator = file!(),
        constructors = messages
            .map(|message| constructor(message, options))
            .join("\n\n"),
        variant_getters = messages
            .map(|message| variant_getter(message, options))
            .join("\n\n"),
        field_getters = messages
            .flat_map(|message| field_getters(message, options))
            .join("\n\n"),
        variant_predicates = messages
            .map(|message| variant_predicate(message, options))
            .join("\n\n")
    )
}

pub(crate) fn codegen(options: &Options) {
    std::fs::write("external/c/bindings_messages.c", contents(options)).unwrap();
}

fn constructor(message: &Message, options: &Options) -> String {
    let initializer_list = message
        .fields
        .map(|field| {
            let unpack = match field.field_type {
                lib_ruby_parser_nodes::MessageFieldType::Str => "UNPACK_StringPtr",
                lib_ruby_parser_nodes::MessageFieldType::Byte => "UNPACK_Byte",
            };
            format!(
                ".{name} = {unpack}({name}_blob)",
                name = field_name(field),
                unpack = unpack
            )
        })
        .join(", ");

    format!(
            "{signature}
{{
    LIB_RUBY_PARSER_{inner_t} inner = {{ {initializer_list} }};
    LIB_RUBY_PARSER_DiagnosticMessage message = {{ .tag = LIB_RUBY_PARSER_MESSAGE_{enum_tag_name}, .as = {{ .{union_variant_name} = inner }} }};
    return PACK_DiagnosticMessage(message);
}}",
            signature = constructor_sig(message, options),
            inner_t = message.camelcase_name,
            initializer_list = initializer_list,
            enum_tag_name = message.upper_name(),
            union_variant_name = message.lower_name()
        )
}
fn variant_getter(message: &Message, options: &Options) -> String {
    format!(
        "{sig}
{{
    const LIB_RUBY_PARSER_DiagnosticMessage *self = (const LIB_RUBY_PARSER_DiagnosticMessage *)self_blob;
    if (self->tag == LIB_RUBY_PARSER_MESSAGE_{tag_name}) {{
        return (const LIB_RUBY_PARSER_{variant_name}_BLOB*)(&(self->as.{union_member}));
    }} else {{
        return NULL;
    }}
}}",
        sig = variant_getter_sig(message, options),
        tag_name = message.upper_name(),
        variant_name = message.camelcase_name,
        union_member = message.lower_name()
    )
}
fn field_getters(message: &Message, options: &Options) -> Vec<String> {
    message.fields.map(|field| {
        format!(
            "{signature}
{{
    const LIB_RUBY_PARSER_{variant_name} *self = (const LIB_RUBY_PARSER_{variant_name} *)self_blob;
    return (const {blob_type} *)(&(self->{field_name}));
}}",
            signature = field_getter_sig(message, field, options),
            variant_name = message.camelcase_name,
            field_name = field_name(field),
            blob_type = blob_type(field),
        )
    })
}
fn variant_predicate(message: &Message, options: &Options) -> String {
    format!(
        "{signature}
{{
    return ((const LIB_RUBY_PARSER_DiagnosticMessage *)self_blob)->tag == LIB_RUBY_PARSER_MESSAGE_{enum_tag_name};
}}",
        signature = variant_predicate_sig(message, options),
        enum_tag_name = message.upper_name()
    )
}
