use crate::codegen::rust::nodes::helpers;
use crate::codegen::rust::nodes::helpers::{filename, node_field_name, struct_name};

use lib_ruby_parser_bindings::helpers::nodes::{
    drop_variant::name as external_drop_variant_name,
    field_getter::name as external_field_getter_name,
    field_setter::name as external_field_setter_name,
    into_internal::name as external_into_internal_name,
};

fn contents(node: &lib_ruby_parser_nodes::Node) -> String {
    format!(
        "// This file is autogenerated by {generator}

{imports}

use super::internal::Internal{struct_name};
use crate::blobs::{{HasBlob, Blob}};

{comment}
#[repr(C)]
pub struct {struct_name} {{
    pub(crate) blob: {blob_name}
}}

impl std::fmt::Debug for {struct_name} {{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {{
        {debug_impl}
    }}
}}

impl PartialEq for {struct_name} {{
    fn eq(&self, other: &Self) -> bool {{
        {partial_eq_impl}
    }}
}}

impl {struct_name} {{
    {getters}

    #[allow(dead_code)]
    pub(crate) fn into_internal(self) -> Internal{struct_name} {{
        let internal = unsafe {{ {external_into_internal_name}(self.blob) }};
        std::mem::forget(self);
        internal
    }}
}}

extern \"C\" {{
    {extern_fns}
}}

impl InnerNode for {struct_name} {{
    fn expression(&self) -> &Loc {{
        self.get_expression_l()
    }}

    fn inspected_children(&self, indent: usize) -> Vec<String> {{
        let mut result = InspectVec::new(indent);
        {inspected_children}
        result.strings()
    }}

    fn str_type(&self) -> &'static str {{
        \"{str_type}\"
    }}

    fn print_with_locs(&self) {{
        println!(\"{{}}\", self.inspect(0));
        {print_with_locs}
    }}
}}

impl Drop for {struct_name} {{
    fn drop(&mut self) {{
        unsafe {{ {external_drop_variant_name}(&mut self.blob) }}
    }}
}}
",
        generator = file!(),
        blob_name = format!("Blob<{}>", struct_name(node)),
        imports = imports(&node).join("\n"),
        comment = node.render_comment("///", 0),
        struct_name = struct_name(node),
        inspected_children = node.fields.map(inspect_field).join("\n        "),
        str_type = node.wqp_name,
        print_with_locs = node.fields.flat_map(print_with_locs).join("\n        "),
        getters = node
            .fields
            .map(|field| getter(node, field))
            .join("\n\n    "),
        external_into_internal_name = external_into_internal_name(node),
        extern_fns = extern_fns(&node).join("\n    "),
        // trait impls
        debug_impl = debug_impl(&node),
        partial_eq_impl = partial_eq_impl(&node),
        external_drop_variant_name = external_drop_variant_name(node),
    )
}

pub(crate) fn codegen(node: &lib_ruby_parser_nodes::Node) {
    let dir = filename(node);
    let path = format!("src/nodes/types/{}/external.rs", dir);
    std::fs::write(&path, contents(node)).unwrap();
}

fn imports(node: &lib_ruby_parser_nodes::Node) -> Vec<&str> {
    let mut imports = vec![];
    imports.push("use crate::nodes::InnerNode;");
    imports.push("use crate::nodes::InspectVec;");
    imports.push("use crate::Loc;");

    let has_field = |field_type: lib_ruby_parser_nodes::NodeFieldType| {
        node.fields.any_field_has_type(field_type)
    };

    if has_field(lib_ruby_parser_nodes::NodeFieldType::Node)
        || has_field(lib_ruby_parser_nodes::NodeFieldType::Nodes)
        || has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeNode {
            regexp_options: true,
        })
        || has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeNode {
            regexp_options: false,
        })
    {
        imports.push("use crate::Node;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::StringValue) {
        imports.push("use crate::Bytes;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeNode {
        regexp_options: true,
    }) || has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeNode {
        regexp_options: false,
    }) {
        imports.push("use crate::containers::ExternalMaybePtr as MaybePtr;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::Node) {
        imports.push("use crate::containers::ExternalPtr as Ptr;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::Nodes) {
        imports.push("use crate::containers::ExternalList as List;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeLoc) {
        imports.push("use crate::containers::ExternalMaybeLoc as MaybeLoc;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::Str { raw: true })
        || has_field(lib_ruby_parser_nodes::NodeFieldType::Str { raw: false })
    {
        imports.push("use crate::containers::ExternalStringPtr as StringPtr;");
    }

    if has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeStr { chars: false })
        || has_field(lib_ruby_parser_nodes::NodeFieldType::MaybeStr { chars: true })
    {
        imports.push("use crate::containers::ExternalMaybeStringPtr as MaybeStringPtr;");
    }

    imports
}

fn inspect_field(field: &lib_ruby_parser_nodes::NodeField) -> String {
    use lib_ruby_parser_nodes::NodeFieldType;

    let method_name = match field.field_type {
        NodeFieldType::Node => "push_node",
        NodeFieldType::Nodes => "push_nodes",
        NodeFieldType::MaybeNode { regexp_options } => {
            if regexp_options {
                "push_regex_options"
            } else if field.always_print {
                "push_maybe_node_or_nil"
            } else {
                "push_maybe_node"
            }
        }
        NodeFieldType::Loc => return format!(""),
        NodeFieldType::MaybeLoc => return format!(""),
        NodeFieldType::Str { raw } => {
            if raw {
                "push_raw_str"
            } else {
                "push_str"
            }
        }
        NodeFieldType::MaybeStr { chars } => {
            if chars {
                "push_chars"
            } else {
                "push_maybe_str"
            }
        }
        NodeFieldType::StringValue => "push_string_value",
        NodeFieldType::U8 => "push_u8",
    };

    format!("result.{}(self.get_{}());", method_name, field.field_name)
}
fn print_with_locs(field: &lib_ruby_parser_nodes::NodeField) -> Vec<String> {
    use lib_ruby_parser_nodes::NodeFieldType;

    match field.field_type {
        NodeFieldType::Node => vec![format!(
            "self.get_{field_name}().inner_ref().print_with_locs();",
            field_name = field.field_name
        )],
        NodeFieldType::Nodes => vec![
            format!(
                "for node in self.get_{field_name}().iter() {{",
                field_name = field.field_name
            ),
            "  node.inner_ref().print_with_locs();".to_string(),
            "}".to_string(),
        ],
        NodeFieldType::MaybeNode { .. } => vec![format!(
            "self.get_{field_name}().as_ref().map(|node| node.inner_ref().print_with_locs());",
            field_name = field.field_name
        )],
        NodeFieldType::Loc => vec![format!(
            "self.get_{field_name}().print(\"{printable_field_name}\");",
            field_name = field.field_name,
            printable_field_name = field
                .field_name
                .strip_suffix("_l")
                .expect("expected loc field to end with _l")
        )],
        NodeFieldType::MaybeLoc => vec![format!(
            "self.get_{field_name}().as_ref().map(|loc| loc.print(\"{printable_field_name}\"));",
            field_name = field.field_name,
            printable_field_name = field
                .field_name
                .strip_suffix("_l")
                .expect("expected loc field to end with _l"),
        )],
        NodeFieldType::Str { .. } => vec![],
        NodeFieldType::MaybeStr { .. } => vec![],
        NodeFieldType::StringValue => vec![],
        NodeFieldType::U8 => vec![],
    }
}

fn getter(node: &lib_ruby_parser_nodes::Node, field: &lib_ruby_parser_nodes::NodeField) -> String {
    format!(
        "/// Returns `{field_name}` field
    pub fn get_{method_name}(&self) -> &{field_type} {{
        unsafe {{
            #[allow(trivial_casts)]
            ({extern_getter}(&self.blob) as *const {field_type})
                .as_ref()
                .unwrap()
        }}
    }}

    /// Sets `{field_name}` field
    pub fn set_{method_name}(&mut self, {field_name}: {field_type}) {{
        unsafe {{ {extern_setter}(&mut self.blob, {field_name}.into_blob()) }}
    }}",
        method_name = field.field_name,
        field_name = node_field_name(field),
        field_type = helpers::field_type(field),
        extern_getter = external_field_getter_name(node, field),
        extern_setter = external_field_setter_name(node, field),
    )
}

fn extern_fns(node: &lib_ruby_parser_nodes::Node) -> Vec<String> {
    let mut result = vec![];

    // Field getters
    {
        node.fields
            .flat_map(|field| {
                let getter = format!(
                    "fn {getter_name}(blob: *const Blob<{node_type}>) -> *mut Blob<{field_type}>;",
                    getter_name = external_field_getter_name(node, field),
                    node_type = struct_name(node),
                    field_type = helpers::field_type(field)
                );

                let setter = format!(
                    "fn {setter_name}(blob: *mut Blob<{node_type}>, blob: Blob<{field_type}>);",
                    setter_name = external_field_setter_name(node, field),
                    node_type = struct_name(node),
                    field_type = helpers::field_type(field)
                );

                vec![getter, setter]
            })
            .into_iter()
            .for_each(|line| result.push(line));
    }

    // into_internal fn
    {
        let line = format!(
            "fn {fn_name}(blob: Blob<{struct_name}>) -> Internal{struct_name};",
            fn_name = external_into_internal_name(node),
            struct_name = struct_name(node)
        );
        result.push(line);
    }

    // drop fn
    {
        let line = format!(
            "fn {fn_name}(blob: *mut Blob<{struct_name}>);",
            fn_name = external_drop_variant_name(node),
            struct_name = struct_name(node),
        );
        result.push(line);
    }

    result
}

fn debug_impl(node: &lib_ruby_parser_nodes::Node) -> String {
    let mut stmts = vec![];
    stmts.push(format!(
        "f.debug_struct(\"{struct_name}\")\n",
        struct_name = struct_name(node)
    ));
    for field in node.fields.0.iter() {
        stmts.push(format!(
            "            .field(\"{field_name}\", &self.get_{method_name}())\n",
            field_name = node_field_name(field),
            method_name = field.field_name
        ));
    }
    stmts.push(format!("            .finish()"));
    stmts.join("")
}
fn partial_eq_impl(node: &lib_ruby_parser_nodes::Node) -> String {
    node.fields
        .map(|field| {
            format!(
                "self.get_{field_name}() == other.get_{field_name}()",
                field_name = field.field_name
            )
        })
        .join("\n            && ")
}
