use crate::codegen::cpp::helpers as cpp_helpers;
use lib_ruby_parser_bindings::{
    helpers::nodes::{
        constructor::sig as external_constructor_sig,
        drop_variant::sig as external_drop_variant_sig,
        field_getter::sig as external_field_getter_sig,
        field_setter::sig as external_field_setter_sig,
        into_internal::sig as external_into_internal_sig,
        into_variant::sig as external_into_variant_sig,
        variant_getter::sig as external_variant_getter_sig,
        variant_predicate::sig as external_variant_predicate_sig,
    },
    Options,
};

fn contents(options: &Options) -> String {
    let nodes = lib_ruby_parser_nodes::nodes();

    format!(
        "// This file is autogenerated by {generator}

#include \"bindings.hpp\"

extern \"C\"
{{
    // Node constructors
    {constructors}

    // Node variant predicates
    {variant_predicates}

    // Node variant getter
    {variant_getters}

    // Node field getters
    {field_getters}

    // Node field setters
    {field_setters}

    // into_variant fns
    {into_variant_fns}

    // into_internal fns
    {into_internal_fns}

    // variant drop fns
    {variant_drop_fns}

    void lib_ruby_parser__external__node__drop(Node_BLOB* self_blob)
    {{
        Node *self = (Node *)self_blob;
        self->~Node();
    }}
}}
",
        generator = file!(),
        // impl
        constructors = nodes.map(|node| constructor(node, options)).join("\n    "),
        variant_predicates = nodes
            .map(|node| variant_predicate(node, options))
            .join("\n    "),
        variant_getters = nodes
            .map(|node| variant_getter(node, options))
            .join("\n    "),
        field_getters = nodes
            .flat_map(|node| field_getters(node, options))
            .join("\n    "),
        field_setters = nodes
            .flat_map(|node| field_setters(node, options))
            .join("\n    "),
        into_internal_fns = nodes
            .map(|node| into_internal_fn(node, options))
            .join("\n    "),
        into_variant_fns = nodes
            .map(|node| into_variant_fn(node, options))
            .join("\n    "),
        variant_drop_fns = nodes
            .map(|node| variant_drop_fn(node, options))
            .join("\n    "),
    )
}

pub(crate) fn codegen(options: &Options) {
    std::fs::write("external/cpp/bindings_nodes.cpp", contents(options)).unwrap();
}

fn constructor(node: &lib_ruby_parser_nodes::Node, options: &Options) -> String {
    let arglist = node
        .fields
        .map(|field| {
            format!(
                "std::move({field_name})",
                field_name = cpp_helpers::nodes::field_name(field),
            )
        })
        .join(", ");

    let unpack_fields = node
        .fields
        .map(|field| unpack_field(field))
        .join("\n        ");

    format!(
        "{sig}
    {{
        {unpack_fields}
        return PACK_Node(Node({variant_name}({arglist})));
    }}",
        sig = external_constructor_sig(node, options),
        unpack_fields = unpack_fields,
        variant_name = node.camelcase_name,
        arglist = arglist
    )
}
fn variant_predicate(node: &lib_ruby_parser_nodes::Node, options: &Options) -> String {
    format!(
        "{sig}
    {{
        const Node *self = (const Node *)self_blob;
        return std::holds_alternative<{variant_name}>(self->variant);
    }}",
        sig = external_variant_predicate_sig(node, options),
        variant_name = node.camelcase_name,
    )
}
fn variant_getter(node: &lib_ruby_parser_nodes::Node, options: &Options) -> String {
    format!(
        "{sig}
    {{
        Node *self = (Node *)self_blob;
        {variant_name} *variant = std::get_if<{variant_name}>(&(self->variant));
        return ({variant_name}_BLOB*)variant;
    }}",
        sig = external_variant_getter_sig(node, options),
        variant_name = node.camelcase_name
    )
}
fn field_getters(node: &lib_ruby_parser_nodes::Node, options: &Options) -> Vec<String> {
    node.fields.map(|field| {
        let field_type = cpp_helpers::nodes::field_type(field);

        format!(
            "{sig}
    {{
        {variant} *self = ({variant} *)self_blob;
        {field_type}* field = &(self->{field_name});
        return ({blob_type} *)field;
    }}",
            sig = external_field_getter_sig(node, field, options),
            variant = node.camelcase_name,
            field_type = field_type,
            field_name = cpp_helpers::nodes::field_name(field),
            blob_type = cpp_helpers::nodes::blob_type(field)
        )
    })
}
fn field_setters(node: &lib_ruby_parser_nodes::Node, options: &Options) -> Vec<String> {
    node.fields.map(|field| {
        format!(
            "{sig}
    {{
        {struct_name}* self = ({struct_name} *)self_blob;
        {unpack_arg}
        self->{field_name} = std::move({field_name});
    }}",
            sig = external_field_setter_sig(node, field, options),
            struct_name = node.camelcase_name,
            field_name = cpp_helpers::nodes::field_name(field),
            unpack_arg = unpack_field(field),
        )
    })
}
fn into_internal_fn(node: &lib_ruby_parser_nodes::Node, options: &Options) -> String {
    let fields = node
        .fields
        .map(|field| {
            let field_name = cpp_helpers::nodes::field_name(field);

            format!(".{field_name} = {field_name}", field_name = field_name,)
        })
        .join(", ");

    let pack_fields = node
        .fields
        .map(|field| pack_field(field))
        .join("\n        ");

    format!(
        "{sig} {{
        {variant_name} self = UNPACK_{variant_name}(self_blob);
        {pack_fields}
        Internal{variant_name} internal = {{ {fields} }};
        return internal;
    }}",
        sig = external_into_internal_sig(node, options),
        variant_name = node.camelcase_name,
        pack_fields = pack_fields,
        fields = fields
    )
}

fn into_variant_fn(node: &lib_ruby_parser_nodes::Node, options: &Options) -> String {
    format!(
        "{sig} {{
        Node self = UNPACK_Node(self_blob);
        {variant_name} variant = std::get<{variant_name}>(std::move(self.variant));
        return PACK_{variant_name}(std::move(variant));
    }}",
        sig = external_into_variant_sig(node, options),
        variant_name = node.camelcase_name,
    )
}

fn variant_drop_fn(node: &lib_ruby_parser_nodes::Node, options: &Options) -> String {
    format!(
        "{sig} {{
        {struct_name} *self = ({struct_name} *)self_blob;
        self->~{struct_name}();
    }}",
        sig = external_drop_variant_sig(node, options),
        struct_name = node.camelcase_name,
    )
}

fn unpack_field(field: &lib_ruby_parser_nodes::NodeField) -> String {
    let field_name = cpp_helpers::nodes::field_name(field);

    match field.field_type {
        lib_ruby_parser_nodes::NodeFieldType::Node => {
            format!(
                "NodePtr {field_name} = std::unique_ptr<Node>((Node *)(UNPACK_Ptr({field_name}_blob).release()));",
                field_name = field_name
            )
        }
        lib_ruby_parser_nodes::NodeFieldType::MaybeNode { .. } => {
            format!(
                "MaybeNodePtr {field_name} = std::unique_ptr<Node>((Node *)(UNPACK_MaybePtr({field_name}_blob).release()));",
                field_name = field_name
            )
        }
        _ => {
            format!(
                "{field_type} {field_name} = {unpack}({field_name}_blob);",
                field_type = cpp_helpers::nodes::field_type(field),
                field_name = field_name,
                unpack = cpp_helpers::nodes::unpack_field_fn(field)
            )
        }
    }
}

fn pack_field(field: &lib_ruby_parser_nodes::NodeField) -> String {
    let field_name = cpp_helpers::nodes::field_name(field);

    match field.field_type {
        lib_ruby_parser_nodes::NodeFieldType::Node => {
            format!(
                "Ptr_BLOB {field_name} = PACK_Ptr(Ptr((int *)(self.{field_name}.release())));",
                field_name = field_name
            )
        }
        lib_ruby_parser_nodes::NodeFieldType::MaybeNode { .. } => {
            format!(
                "MaybePtr_BLOB {field_name} = PACK_MaybePtr(MaybePtr((int *)(self.{field_name}.release())));",
                field_name = field_name
            )
        }
        _ => {
            format!(
                "{field_type}_BLOB {field_name} = {pack}(std::move(self.{field_name}));",
                field_type = cpp_helpers::nodes::field_type(field),
                field_name = field_name,
                pack = cpp_helpers::nodes::pack_field_fn(field)
            )
        }
    }
}
