use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


/// Represents an executable string (i.e. `` `sh #{script_name}` ``)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Xstr {
    /// A list of string parts (static literals and interpolated expressions)
    ///
    pub parts: List<Node>,
    /// Location of the string begin
    ///
    /// ```text
    /// `#{foo}`
    /// ~
    ///
    /// %X{#{foo}}
    /// ~~~
    /// ```
    ///
    pub begin_l: Loc,
    /// Location of the string end
    ///
    /// ```text
    /// `#{foo}`
    ///        ~
    ///
    /// %X{#{foo}}
    ///          ~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// `#{foo}`
    /// ~~~~~~~~
    ///
    /// %X{#{foo}}
    /// ~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Xstr {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.parts);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "xstr"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.parts.iter() {
            node.inner_ref().print_with_locs();
        }
        self.begin_l.print("begin");
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

