use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


/// Represents an `undef` keyword (e.g. `undef foo, :bar`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Undef {
    /// A list of names to `undef`
    ///
    pub names: List<Node>,
    /// Location the `undef` keyword
    ///
    /// ```text
    /// undef foo, :bar
    /// ~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// undef :foo, bar
    /// ~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Undef {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.names);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "undef"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.names.iter() {
            node.inner_ref().print_with_locs();
        }
        self.keyword_l.print("keyword");
        self.expression_l.print("expression");
    }
}

