use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents a `rescue` block
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Rescue {
    /// Body of the block that is wrapped into `rescue` (i.e. the part that may throw an error)
    ///
    pub body: MaybePtr<Node>,
    /// A list of `rescue` handlers (see `RescueBody` node)
    ///
    pub rescue_bodies: List<Node>,
    /// Else branch.
    ///
    /// `None` if there's no `else` branch
    ///
    pub else_: MaybePtr<Node>,
    /// Location of the `else` keyword
    ///
    /// ```text
    /// begin; 1; rescue StandardError => e; 2; else; 3; end
    ///                                         ~~~~
    /// ```
    ///
    /// `None` if there's no `else` branch
    ///
    pub else_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// begin; 1; rescue StandardError => e; 2; else; 3; end
    ///        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    /// ```
    ///
    /// **Note**: `begin/end` keywords belong to `KwBegin` node
    ///
    pub expression_l: Loc,
}

impl InnerNode for Rescue {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node_or_nil(&self.body);
        result.push_nodes(&self.rescue_bodies);
        result.push_maybe_node_or_nil(&self.else_);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "rescue"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        for node in self.rescue_bodies.iter() {
            node.inner_ref().print_with_locs();
        }
        self.else_.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.else_l.as_ref().map(|loc| loc.print("else"));
        self.expression_l.print("expression");
    }
}

