use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents rational literal (e.g. `1r`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Rational {
    /// String value of the literal, `String("1r")` for `1r`
    ///
    pub value: StringPtr,
    /// Location of the unary `-` (but not `+`)
    ///
    /// ```text
    /// -1r
    /// ~
    /// ```
    ///
    pub operator_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// -1r
    /// ~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Rational {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_str(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "rational"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.operator_l.as_ref().map(|loc| loc.print("operator"));
        self.expression_l.print("expression");
    }
}

