use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


/// Represents `END { .. }` statement
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Postexe {
    /// Body of the block
    ///
    pub body: MaybePtr<Node>,
    /// Location of the `END` keyword
    ///
    /// ```text
    /// END { 42 }
    /// ~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the open parenthesis
    ///
    /// ```text
    /// END { 42 }
    ///     ~
    /// ```
    ///
    pub begin_l: Loc,
    /// Location of the closing parenthesis
    ///
    /// ```text
    /// END { 42 }
    ///          ~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// END { 42 }
    /// ~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Postexe {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "postexe"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.begin_l.print("begin");
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

