use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents `lhs ||= rhs` assignment
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct OrAsgn {
    /// Left hand statement
    ///
    pub recv: Ptr<Node>,
    /// Right hand statement
    ///
    pub value: Ptr<Node>,
    /// Location of the `||=` operator
    ///
    /// ```text
    /// foo ||= bar
    ///     ~~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo ||= bar
    /// ~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for OrAsgn {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.recv);
        result.push_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "or_asgn"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.recv.inner_ref().print_with_locs();
        self.value.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

