use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents `foo || bar` (or `foo or bar`) statement.
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Or {
    /// Left hand statement
    ///
    pub lhs: Ptr<Node>,
    /// Right hand statement
    ///
    pub rhs: Ptr<Node>,
    /// Location of the `||`/`or` operator
    ///
    /// ```text
    /// foo || bar
    ///     ~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo || bar
    /// ~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Or {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.lhs);
        result.push_node(&self.rhs);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "or"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.lhs.inner_ref().print_with_locs();
        self.rhs.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

