use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents an operation with assignment (e.g. `a += 1`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct OpAsgn {
    /// Left hand statement of the assignment
    ///
    pub recv: Ptr<Node>,
    /// Operator, can be one of:
    ///     1. `+=`
    ///     2. `-=`
    ///     3. `*=`
    ///     4. `/=`
    ///     5. `|=`
    ///     6. `&=`
    ///     7. `>>=`
    ///     8. `<<=`
    ///     9. `%=`
    ///     10. `^=`
    ///     11. `**=`
    ///
    pub operator: StringPtr,
    /// Right hand statement of the assignment
    ///
    pub value: Ptr<Node>,
    /// Location of the operator
    ///
    /// ```text
    /// a.b <<= c
    ///     ~~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the operator
    ///
    /// ```text
    /// a.b <<= c
    /// ~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for OpAsgn {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.recv);
        result.push_str(&self.operator);
        result.push_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "op_asgn"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.recv.inner_ref().print_with_locs();
        self.value.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

