use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents a block that takes numbered parameters (i.e. `proc { _1 }`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Numblock {
    /// Method call that takes a block
    ///
    pub call: Ptr<Node>,
    /// Number of parameters that block takes
    ///
    pub numargs: u8,
    /// Block body
    ///
    pub body: Ptr<Node>,
    /// Location of the open brace
    ///
    /// ```text
    /// proc { _1 }
    ///      ~
    /// ```
    ///
    pub begin_l: Loc,
    /// Location of the closing brace
    ///
    /// ```text
    /// proc { _1 }
    ///           ~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the open brace
    ///
    /// ```text
    /// proc { _1 }
    /// ~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Numblock {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.call);
        result.push_u8(&self.numargs);
        result.push_node(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "numblock"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.call.inner_ref().print_with_locs();
        self.body.inner_ref().print_with_locs();
        self.begin_l.print("begin");
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

