use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


/// Represents a wildcard pattern used in pattern matching (i.e. `in *foo`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct MatchRest {
    /// Name of the variable name
    ///
    /// `None` if there's no name (i.e. `in *`)
    ///
    pub name: MaybePtr<Node>,
    /// Location of the `*` operator
    ///
    /// ```text
    /// case foo; in *bar; end
    ///              ~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the `*` operator
    ///
    /// ```text
    /// case foo; in *bar; end
    ///              ~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for MatchRest {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node(&self.name);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "match_rest"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.name.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

