use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents implicit matching using `if /regex/`
///
/// ```text
/// if /.*/
///   puts 'true'
/// else
///   puts 'false'
/// end
/// ```
/// Prints "false".
///
/// Under the hood this construction matches regex against `$_`, so the following works:
/// ```text
/// $_ = 'match_me'
/// if /match_me/
///   puts 'true'
/// else
///   puts 'false'
/// end
/// ```
/// this code prints "true".
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct MatchCurrentLine {
    /// Given regex
    ///
    pub re: Ptr<Node>,
    /// Location of the regex
    ///
    /// ```text
    /// if /re/; end
    ///    ~~~~
    /// ```
    ///
    /// Technically this location is redundant, but keeping it is the only way to
    /// have the same interface for all nodes.
    ///
    pub expression_l: Loc,
}

impl InnerNode for MatchCurrentLine {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.re);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "match_current_line"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.re.inner_ref().print_with_locs();
        self.expression_l.print("expression");
    }
}

