use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents local variable assignment (i.e. `foo = 42`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Lvasgn {
    /// Name of the local variable
    ///
    pub name: StringPtr,
    /// Value that is assigned to a local variable
    ///
    pub value: MaybePtr<Node>,
    /// Location of the local variable name
    ///
    /// ```text
    /// foo = 42
    /// ~~~
    /// ```
    ///
    pub name_l: Loc,
    /// Location of the `=` operator
    ///
    /// ```text
    /// foo = 42
    ///     ~
    /// ```
    ///
    /// `None` if local variable assignment is a part of the multi-assignment.
    /// In such case `value` is a part of the `Masgn` node.
    ///
    pub operator_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo = 42
    /// ~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Lvasgn {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_str(&self.name);
        result.push_maybe_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "lvasgn"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.value.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.name_l.print("name");
        self.operator_l.as_ref().map(|loc| loc.print("operator"));
        self.expression_l.print("expression");
    }
}

