use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents an explicit `begin; end` block.
///
/// The reason why it's different is that only
/// ```text
/// begin; foo; end while cond
/// ```
/// is a post-while loop (same with post-until loop)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct KwBegin {
    /// A list of statements
    ///
    pub statements: List<Node>,
    /// Location of the `begin` keyword
    ///
    /// ```text
    /// begin; foo; end
    /// ~~~~~
    /// ```
    ///
    pub begin_l: MaybeLoc,
    /// Location of the `begin` keyword
    ///
    /// ```text
    /// begin; foo; end
    ///             ~~~
    /// ```
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// begin; foo; bar
    /// ~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for KwBegin {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.statements);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "kwbegin"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.statements.iter() {
            node.inner_ref().print_with_locs();
        }
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

