use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


/// Represents kwargs that are given to a method call, super or yield (i.e. `foo(bar: 1)`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Kwargs {
    /// A list of key-value pairs
    ///
    pub pairs: List<Node>,
    /// Location of the full expression
    ///
    /// ```text
    /// foo(bar: 1)
    ///     ~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Kwargs {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.pairs);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "kwargs"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.pairs.iter() {
            node.inner_ref().print_with_locs();
        }
        self.expression_l.print("expression");
    }
}

