use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents access to instance variable (i.e. `@foo`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Ivar {
    /// Name of the instance variable, `String("@foo")` in `@foo`
    ///
    pub name: StringPtr,
    /// Location of the full expression
    ///
    /// ```text
    /// @foo
    /// ~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Ivar {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_str(&self.name);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "ivar"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.expression_l.print("expression");
    }
}

