use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


/// Represents inclusive flip-flop (i.e. in `if foo..bar; end`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct IFlipFlop {
    /// Left part of the flip-flop. `None` if based on a range without begin (`..bar`)
    ///
    pub left: MaybePtr<Node>,
    /// Right part of the flip-flop. `None` if based on a range without end (`foo..`)
    ///
    pub right: MaybePtr<Node>,
    /// Location of the `..` operator
    ///
    /// ```text
    /// if foo..bar; end
    ///       ~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// if foo..bar; end
    ///    ~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for IFlipFlop {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node_or_nil(&self.left);
        result.push_maybe_node_or_nil(&self.right);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "iflipflop"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.left.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.right.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

