use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents an `if` statement (i.e. `if foo; bar; else; baz; end`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct If {
    /// Condition given to the `if` statement, `Lvar("a")` for `if a; b; else; c; end`
    ///
    pub cond: Ptr<Node>,
    /// True-branch of the `if` statement, `Lvar("b")` for `if a; b; else; c; end`
    ///
    pub if_true: MaybePtr<Node>,
    /// False-branch of the `if` statement, `Lvar("c")` for `if a; b; else; c; end`
    ///
    pub if_false: MaybePtr<Node>,
    /// Location of the `if` keyword
    ///
    /// ```text
    /// if foo; end
    /// ~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `then` keyword
    ///
    /// ```text
    /// if foo then; end
    ///        ~~~~
    /// ```
    ///
    /// `None` if `then` keyword is omitted
    ///
    pub begin_l: Loc,
    /// Location of the `else` keyword
    ///
    /// ```text
    /// if foo; else; end
    ///         ~~~~
    /// ```
    ///
    /// `None` if there's no `else` branch
    ///
    pub else_l: MaybeLoc,
    /// Location of the `end` keyword
    ///
    /// ```text
    /// if foo; end
    ///         ~~~
    /// ```
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// if a then; b; else; c end
    /// ~~~~~~~~~~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for If {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.cond);
        result.push_maybe_node_or_nil(&self.if_true);
        result.push_maybe_node_or_nil(&self.if_false);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "if"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.cond.inner_ref().print_with_locs();
        self.if_true.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.if_false.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.begin_l.print("begin");
        self.else_l.as_ref().map(|loc| loc.print("else"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

