use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents a singleton method definition (i.e. `def self.foo; end`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Defs {
    /// Definee of a method definition, `Lvar("x")` for `def x.foo; end
    ///
    pub definee: Ptr<Node>,
    /// Name of the method, `String("foo")` for `def x.foo; end`
    ///
    pub name: StringPtr,
    /// Arguments of a method, `None` if there's no arguments.
    ///
    /// All information about parentheses around arguments is stored in this node.
    ///
    pub args: MaybePtr<Node>,
    /// Body of the method, `None` if there's no body.
    ///
    pub body: MaybePtr<Node>,
    /// Location of the `def` keyword
    ///
    /// ```text
    /// def self.foo; end
    /// ~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `.`
    ///
    /// ```text
    /// def self.foo; end
    ///         ~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the method name
    ///
    /// ```text
    /// def self.foo; end
    ///          ~~~
    /// ```
    ///
    pub name_l: Loc,
    /// Location of the `=` operator for endless method definition
    ///
    /// ```text
    /// def self.foo() = 42
    ///                ~
    /// ```
    ///
    /// `None` for regular method definition
    ///
    pub assignment_l: MaybeLoc,
    /// Location of the `end` keyword
    ///
    /// ```text
    /// def self.foo; end
    ///               ~~~
    /// ```
    ///
    /// `None` for endless method definition
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// def self.foo; end
    /// ~~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Defs {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.definee);
        result.push_str(&self.name);
        result.push_maybe_node_or_nil(&self.args);
        result.push_maybe_node_or_nil(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "defs"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.definee.inner_ref().print_with_locs();
        self.args.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.operator_l.print("operator");
        self.name_l.print("name");
        self.assignment_l.as_ref().map(|loc| loc.print("assignment"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

