use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents conditional method call using `&.` operator
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct CSend {
    /// Receiver of the method call, `Int("1")` for `1&.foo`
    ///
    pub recv: Ptr<Node>,
    /// Name of the method, `String("foo")` for `1&.foo`
    ///
    pub method_name: StringPtr,
    /// List of arguments
    ///
    /// ```text
    /// foo&.bar(42)
    /// # and also setters like
    /// foo&.bar = 42
    /// ```
    ///
    pub args: List<Node>,
    /// Location of the `&.` operator
    ///
    /// ```text
    /// foo&.bar
    ///    ~~
    /// ```
    ///
    pub dot_l: Loc,
    /// Location of the method name
    ///
    /// ```text
    /// foo&.bar(42)
    ///      ~~~
    /// ```
    ///
    /// `None` in a very special case when method call is implicit (i.e. `foo&.()`)
    ///
    pub selector_l: MaybeLoc,
    /// Location of the open parenthesis
    ///
    /// ```text
    /// foo&.bar(42)
    ///         ~
    /// ```
    ///
    /// `None` if there are no parentheses
    ///
    pub begin_l: MaybeLoc,
    /// Location of the closing parenthesis
    ///
    /// ```text
    /// foo&.bar(42)
    ///            ~
    /// ```
    ///
    /// `None` if there are no parentheses
    ///
    pub end_l: MaybeLoc,
    /// Location of the operator if `CSend` is a part of assignment like
    ///
    /// ```text
    /// foo&.bar = 1
    ///          ~
    /// ```
    ///
    /// `None` for a regular call.
    ///
    pub operator_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo&.bar(42)
    /// ~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for CSend {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.recv);
        result.push_str(&self.method_name);
        result.push_nodes(&self.args);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "csend"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.recv.inner_ref().print_with_locs();
        for node in self.args.iter() {
            node.inner_ref().print_with_locs();
        }
        self.dot_l.print("dot");
        self.selector_l.as_ref().map(|loc| loc.print("selector"));
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.operator_l.as_ref().map(|loc| loc.print("operator"));
        self.expression_l.print("expression");
    }
}

