use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents constant access (i.e. `Foo::Bar`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Const {
    /// Scope where the constant is taken from:
    ///     1. `Some(Const("A"))` for `A::B`
    ///     2. `None` if it's taken from the current scope (i.e. `A`)
    ///     3. `Some(Cbase)` if it's taken from the global scope (i.e. `::A`)
    ///
    pub scope: MaybePtr<Node>,
    /// Name of the constant, `String("Foo")` for `Foo`
    ///
    pub name: StringPtr,
    /// Location of the `::` operator. `None` if constant is taken from the current scope.
    ///
    /// ```text
    /// A::B
    ///  ~~
    /// ```
    ///
    pub double_colon_l: MaybeLoc,
    /// Location of the constant name
    ///
    /// ```text
    /// Foo::Bar
    ///      ~~~
    /// ```
    ///
    pub name_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// Foo::Bar
    /// ~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Const {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node_or_nil(&self.scope);
        result.push_str(&self.name);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "const"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.scope.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.double_colon_l.as_ref().map(|loc| loc.print("double_colon"));
        self.name_l.print("name");
        self.expression_l.print("expression");
    }
}

