use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents a `case` statement used for pattern matching (for regular `case` see `Case` node)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct CaseMatch {
    /// Expression given to `case`, `Int("1")` for `case 1; in 1; end`
    /// `None` for code like
    ///
    /// ```text
    /// case
    /// in pattern
    /// end
    /// ```
    ///
    pub expr: Ptr<Node>,
    /// A list of `InPattern` nodes (each has `pattern`, `guard` and `body`)
    ///
    pub in_bodies: List<Node>,
    /// Body of the `else` branch, `None` if there's no `else` branch
    ///
    pub else_body: MaybePtr<Node>,
    /// Location of the `case` keyword
    ///
    /// ```text
    /// case 1; in 2; end
    /// ~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `else` keyword
    ///
    /// ```text
    /// case 1; in 2; else; end
    ///               ~~~~
    /// ```
    ///
    /// `None` if there's no `else` branch
    ///
    pub else_l: MaybeLoc,
    /// Location of the `end` keyword
    ///
    /// ```text
    /// case 1; in 2; end
    ///               ~~~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// case 1; in 2; end
    /// ~~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for CaseMatch {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.expr);
        result.push_nodes(&self.in_bodies);
        result.push_maybe_node_or_nil(&self.else_body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "case_match"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.expr.inner_ref().print_with_locs();
        for node in self.in_bodies.iter() {
            node.inner_ref().print_with_locs();
        }
        self.else_body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.else_l.as_ref().map(|loc| loc.print("else"));
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

