use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents a `&blk` argument of the method call (but not of the method definition, see `BlockArg`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct BlockPass {
    /// Value that is converted to a block
    ///
    /// `Int("1")` in `foo(&1)` (yes, it's possible)
    ///
    pub value: Ptr<Node>,
    /// Location of the `&` operator
    ///
    /// ```text
    /// foo(&blk)
    ///     ~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo(&bar)
    ///     ~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for BlockPass {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "block_pass"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.value.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

