use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents `a &&= 1` statement.
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct AndAsgn {
    /// Receiver of the `&&=` operation.
    ///
    /// `Lvasgn("a")` node for `a &&= 1`
    ///
    pub recv: Ptr<Node>,
    /// Right hand statement of assignment
    ///
    /// `Int("1")` node for `a &&= 1`
    ///
    pub value: Ptr<Node>,
    /// Location of the `&&=` operator
    ///
    /// ```text
    /// a &&= 1
    ///   ~~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// a &&= 1
    /// ~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for AndAsgn {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.recv);
        result.push_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "and_asgn"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.recv.inner_ref().print_with_locs();
        self.value.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

