use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents a post-while loop
///
/// ```text
/// begin
///   foo
/// end while bar
/// ```
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct WhilePost {
    /// Condition of the loop
    ///
    pub cond: Ptr<Node>,
    /// Body of the loop
    ///
    pub body: Ptr<Node>,
    /// Location of the `while` keyword
    ///
    /// ```text
    /// begin; foo; end while bar
    ///                 ~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `while` keyword
    ///
    /// ```text
    /// begin; foo; end while bar
    /// ~~~~~~~~~~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for WhilePost {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.cond);
        result.push_node(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "while_post"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.cond.inner_ref().print_with_locs();
        self.body.inner_ref().print_with_locs();
        self.keyword_l.print("keyword");
        self.expression_l.print("expression");
    }
}

