use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents `while` loop
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct While {
    /// Condition of the loop
    ///
    pub cond: Ptr<Node>,
    /// Body of the loop.
    ///
    /// `None` if body is empty
    ///
    pub body: MaybePtr<Node>,
    /// Location of the `while` keyword
    ///
    /// ```text
    /// while cond do; foo; end
    /// ~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `do` keyword
    ///
    /// ```text
    /// while cond do; foo; end
    ///            ~~
    /// ```
    ///
    /// `do` is optional, and so `begin_l` can be `None`
    ///
    pub begin_l: MaybeLoc,
    /// Location of the `end` keyword
    ///
    /// ```text
    /// while cond do; foo; end
    ///                     ~~~
    /// ```
    ///
    /// `None` if loop is a modifier (i.e. `foo while bar`)
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// while cond do; foo; end
    /// ~~~~~~~~~~~~~~~~~~~~~~~
    ///
    /// foo while bar
    /// ~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for While {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.cond);
        result.push_maybe_node_or_nil(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "while"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.cond.inner_ref().print_with_locs();
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

