use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::StringValue;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents a plain non-interpolated string literal (e.g. `"foo"`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Str {
    /// Value of the string literal
    ///
    /// Note that it's a `StringValue`, not a `String`.
    /// The reason is that you can get UTF-8 incompatible strings
    /// from a valid UTF-8 source using escape sequences like `"\xFF"`
    ///
    /// These "\", "x", "F", "F" chars are valid separately, but together
    /// they construct a char with code = 255 that is invalid for UTF-8.
    ///
    /// You can use `to_string_lossy` or `to_string` methods to get a raw string value.
    ///
    pub value: StringValue,
    /// Location of the string begin
    ///
    /// ```text
    /// "foo"
    /// ~
    /// ```
    ///
    /// `None` if string literal is a part of the words array (like `%w[foo bar baz]`)
    ///
    pub begin_l: MaybeLoc,
    /// Location of the string begin
    ///
    /// ```text
    /// "foo"
    ///     ~
    /// ```
    ///
    /// `None` if string literal is a part of the words array (like `%w[foo bar baz]`)
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// "foo"
    /// ~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Str {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_string_value(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "str"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

