use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents a method call (e.g. `foo.bar(42)`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Send {
    /// Receiver of the method call
    ///
    /// `None` for implicit method call (e.g. `foo(42)`)
    ///
    pub recv: MaybePtr<Node>,
    /// Name of the method that is called
    ///
    pub method_name: StringPtr,
    /// A list of arguments
    ///
    pub args: List<Node>,
    /// Location of the `.` operator
    ///
    /// ```text
    /// foo.bar(42)
    ///    ~
    /// ```
    ///
    /// `None` for implicit method call (e.g. `foo(42)`)
    ///
    pub dot_l: MaybeLoc,
    /// Location of the method name
    ///
    /// ```text
    /// foo.bar(42)
    ///     ~~~
    /// ```
    ///
    /// `None` in a very special case when method call is implicit (i.e. `foo.(42)`)
    ///
    pub selector_l: MaybeLoc,
    /// Location of open parenthesis
    ///
    /// ```text
    /// foo(42)
    ///    ~
    /// ```
    ///
    /// `None` if there are parentheses
    ///
    pub begin_l: MaybeLoc,
    /// Location of closing parenthesis
    ///
    /// ```text
    /// foo(42)
    ///       ~
    /// ```
    ///
    /// `None` if there are parentheses
    ///
    pub end_l: MaybeLoc,
    /// Location of the operator if method is a setter
    ///
    /// ```text
    /// foo.bar = 42
    ///         ~
    /// ```
    ///
    /// `None` otherwise
    ///
    pub operator_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo.bar(42)
    /// ~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Send {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node_or_nil(&self.recv);
        result.push_str(&self.method_name);
        result.push_nodes(&self.args);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "send"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.recv.as_ref().map(|node| node.inner_ref().print_with_locs());
        for node in self.args.iter() {
            node.inner_ref().print_with_locs();
        }
        self.dot_l.as_ref().map(|loc| loc.print("dot"));
        self.selector_l.as_ref().map(|loc| loc.print("selector"));
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.operator_l.as_ref().map(|loc| loc.print("operator"));
        self.expression_l.print("expression");
    }
}

