use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents opening a singleton class (i.e. `class << foo; ... end;`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct SClass {
    /// Expression that is used to get a singleton class
    ///
    /// `Lvar("foo")` for `class << foo; end`
    ///
    pub expr: Ptr<Node>,
    /// Body of the block
    ///
    pub body: MaybePtr<Node>,
    /// Location of the `class` keyword
    ///
    /// ```text
    /// class << foo; end
    /// ~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `<<` operator
    ///
    /// ```text
    /// class << foo; end
    ///       ~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the `end` keyword
    ///
    /// ```text
    /// class << foo; end
    ///               ~~~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// class << foo; end
    /// ~~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for SClass {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.expr);
        result.push_maybe_node_or_nil(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "sclass"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.expr.inner_ref().print_with_locs();
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.operator_l.print("operator");
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

