use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybeStringPtr = ExternalMaybeStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeStringPtr = Option<String>;


/// Represents positional rest argument (i.e. `*foo` in `def m(*foo); end`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Restarg {
    /// Name of the argument.
    ///
    /// `None` if argument has no name (i.e. `def m(*); end`)
    ///
    pub name: MaybeStringPtr,
    /// Location of the `*` operator
    ///
    /// ```text
    /// def m(*foo); end
    ///       ~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the argument name
    ///
    /// ```text
    /// def m(*foo); end
    ///        ~~~
    /// ```
    ///
    pub name_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// def m(*foo); end
    ///       ~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Restarg {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_str(&self.name);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "restarg"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.operator_l.print("operator");
        self.name_l.as_ref().map(|loc| loc.print("name"));
        self.expression_l.print("expression");
    }
}

