use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


/// Represents regex literal (e.g. `/foo/`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Regexp {
    /// A list of static and dynamic regex parts
    ///
    pub parts: List<Node>,
    /// Regex options.
    ///
    /// `None` if regex has no explicit flags
    ///
    pub options: MaybePtr<Node>,
    /// Location of the regex begin
    ///
    /// ```text
    /// /foo/
    /// ~
    ///
    /// %r{foo}
    /// ~~
    /// ```
    ///
    pub begin_l: Loc,
    /// Location of the regex end
    ///
    /// ```text
    /// /foo/
    ///     ~
    ///
    /// %r{foo}
    ///       ~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// /foo/mix
    /// ~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Regexp {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.parts);
        result.push_regex_options(&self.options);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "regexp"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.parts.iter() {
            node.inner_ref().print_with_locs();
        }
        self.options.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.begin_l.print("begin");
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

