use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents numeric global variable (e.g. `$1`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct NthRef {
    /// Name of the variable, `String("1")` for `$1`
    ///
    pub name: StringPtr,
    /// Location of the full expression
    ///
    /// ```text
    /// $1
    /// ~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for NthRef {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_raw_str(&self.name);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "nth_ref"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.expression_l.print("expression");
    }
}

