use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents module declaration using `module` keyword
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Module {
    /// Name of the module
    ///
    pub name: Ptr<Node>,
    /// Body of the module
    ///
    /// `None` if module has no body
    ///
    pub body: MaybePtr<Node>,
    /// Location of the `module` keyword
    ///
    /// ```text
    /// module M; end
    /// ~~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `end` keyword
    ///
    /// ```text
    /// module M; end
    ///           ~~~
    /// ```
    ///
    pub end_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// module M; end
    /// ~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Module {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.name);
        result.push_maybe_node_or_nil(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "module"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.name.inner_ref().print_with_locs();
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.end_l.print("end");
        self.expression_l.print("expression");
    }
}

