use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents a one-line pattern matching that never throws but returns true/false (i.e. `foo in pattern`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct MatchPatternP {
    /// Value that is used for matching
    ///
    pub value: Ptr<Node>,
    /// Pattern that is used for matching
    ///
    pub pattern: Ptr<Node>,
    /// Location of the `in` operator
    ///
    /// ```text
    /// foo in pattern
    ///     ~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo in pattern
    /// ~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for MatchPatternP {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.value);
        result.push_node(&self.pattern);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "match_pattern_p"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.value.inner_ref().print_with_locs();
        self.pattern.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

