use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents a keyword arguments splat (i.e. `**bar` in a call like `foo(**bar)`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Kwsplat {
    /// Value that is converted into a `Hash` using `**`
    ///
    pub value: Ptr<Node>,
    /// Location of the `**` operator
    ///
    /// ```text
    /// foo(**bar)
    ///     ~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// foo(**bar)
    ///     ~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Kwsplat {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "kwsplat"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.value.inner_ref().print_with_locs();
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

