use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents an `in pattern` branch of the pattern matching
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct InPattern {
    /// Value that is used for matching
    ///
    pub pattern: Ptr<Node>,
    /// Guard that is used for matching
    ///
    /// Optional, so can be `None`
    ///
    pub guard: MaybePtr<Node>,
    /// Body of the branch that is invoked if value matches pattern
    ///
    pub body: MaybePtr<Node>,
    /// Location of the `in` keyword
    ///
    /// ```text
    /// case value; in pattern; end
    ///             ~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the `then` keyword
    ///
    /// ```text
    /// case value; in pattern then; end
    ///                        ~~~~
    /// ```
    ///
    pub begin_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// case value; in pattern then; 42; end
    ///             ~~~~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for InPattern {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.pattern);
        result.push_maybe_node_or_nil(&self.guard);
        result.push_maybe_node_or_nil(&self.body);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "in_pattern"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.pattern.inner_ref().print_with_locs();
        self.guard.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.body.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.begin_l.print("begin");
        self.expression_l.print("expression");
    }
}

