use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents an `if`/`unless` modifier (i.e. `stmt if cond`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct IfMod {
    /// Condition of the modifier
    ///
    pub cond: Ptr<Node>,
    /// True-branch of the modifier.
    ///
    /// Always set for `if` modifier.
    /// Always `None` for `unless` modifier.
    ///
    pub if_true: MaybePtr<Node>,
    /// False-branch of the modifier.
    ///
    /// Always set for `unless` modifier.
    /// Always `None` for `if` modifier.
    ///
    pub if_false: MaybePtr<Node>,
    /// Location of the `if`/`unless` keyword
    ///
    /// ```text
    /// stmt if cond
    ///      ~~
    ///
    /// stmt unless cond
    ///      ~~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// stmt if cond
    /// ~~~~~~~~~~~~
    ///
    /// stmt unless cond
    /// ~~~~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for IfMod {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.cond);
        result.push_maybe_node_or_nil(&self.if_true);
        result.push_maybe_node_or_nil(&self.if_false);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "if"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.cond.inner_ref().print_with_locs();
        self.if_true.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.if_false.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.keyword_l.print("keyword");
        self.expression_l.print("expression");
    }
}

