use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents a hash literal (i.e. `{ foo: 42 }`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Hash {
    /// A list of key-value pairs
    ///
    pub pairs: List<Node>,
    /// Location of the open parenthesis
    ///
    /// ```text
    /// { a: 1 }
    /// ~
    /// ```
    ///
    /// `None` if hash literal is implicit, e.g. `foo(key: "value")`
    ///
    pub begin_l: MaybeLoc,
    /// Location of the closing parenthesis
    ///
    /// ```text
    /// { a: 1 }
    ///        ~
    /// ```
    ///
    /// `None` if hash literal is implicit, e.g. `foo(key: "value")`
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// { a: 1 }
    /// ~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Hash {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.pairs);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "hash"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.pairs.iter() {
            node.inner_ref().print_with_locs();
        }
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

