use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


/// Represents range literal with excluded `end` (i.e. `1...3`)
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Erange {
    /// Begin of the range, `None` if range has no begin (i.e `...42`)
    ///
    pub left: MaybePtr<Node>,
    /// End of the range, `None` if range has no end (i.e `42...`)
    ///
    pub right: MaybePtr<Node>,
    /// Location of the `...` operator
    ///
    /// ```text
    /// 1...3
    ///  ~~~
    /// ```
    ///
    pub operator_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// 1...3
    /// ~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Erange {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node_or_nil(&self.left);
        result.push_maybe_node_or_nil(&self.right);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "erange"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.left.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.right.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.operator_l.print("operator");
        self.expression_l.print("expression");
    }
}

