use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents a symbol with interpolation (i.e. `:"#{foo}"`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Dsym {
    /// A list of symbol parts (static literals and interpolated expressions)
    ///
    pub parts: List<Node>,
    /// Location of the symbol begin
    ///
    /// ```text
    /// :"#{foo}"
    /// ~~
    /// ```
    ///
    /// `None` if `Dsym` is a part of the interpolated symbol array:
    ///
    /// ```text
    /// %I[#{bar}]
    /// ```
    ///
    pub begin_l: MaybeLoc,
    /// Location of the symbol begin
    ///
    /// ```text
    /// :"#{foo}"
    ///         ~
    /// ```
    ///
    /// `None` if `Dsym` is a part of the interpolated symbol array:
    ///
    /// ```text
    /// %I[#{bar}]
    /// ```
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// :"#{foo}"
    /// ~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Dsym {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.parts);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "dsym"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.parts.iter() {
            node.inner_ref().print_with_locs();
        }
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

