use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybePtr;
#[cfg(feature = "compile-with-external-structures")]
type MaybePtr<T> = ExternalMaybePtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybePtr<T> = Option<Box<T>>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents a constant assignment (i.e. `A = 1`)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Casgn {
    /// Scope where the constant is defined:
    ///     1. `Some(Const("A"))` for `A::B = 1`
    ///     2. `None` if it's defined in the current scope (i.e. `A = 1`)
    ///     3. `Some(Cbase)` if it's defined in the global scope (i.e. `::A = 1`)
    ///
    pub scope: MaybePtr<Node>,
    /// Name of the constant, `String("A")` for `A = 1`
    ///
    pub name: StringPtr,
    /// Value that is assigned to a constant, `Int("1")` for `A = 1`.
    ///
    /// **Note**: `None` if constant assignment is a part of the multi-assignment.
    /// In such case `value` belongs to `Masgn` node of the multi-assignment.
    ///
    pub value: MaybePtr<Node>,
    /// Location of the `::` operator
    ///
    /// ```text
    /// A::B = 1
    ///  ~~
    ///
    /// ::A = 1
    /// ~~
    /// ```
    ///
    /// `None` if the constant is defined in the current scope
    ///
    pub double_colon_l: MaybeLoc,
    /// Location of the constant name
    ///
    /// ```text
    /// A::CONST = 1
    ///    ~~~~~
    /// ```
    ///
    pub name_l: Loc,
    /// Location of the `=` operator
    ///
    /// ```text
    /// A = 1
    ///   ~
    /// ```
    ///
    /// `None` if constant assignment is a part of the multi-assignment.
    /// In such case `=` belongs to a `Masgn` node
    ///
    pub operator_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// A = 1
    /// ~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Casgn {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_maybe_node_or_nil(&self.scope);
        result.push_str(&self.name);
        result.push_maybe_node(&self.value);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "casgn"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.scope.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.value.as_ref().map(|node| node.inner_ref().print_with_locs());
        self.double_colon_l.as_ref().map(|loc| loc.print("double_colon"));
        self.name_l.print("name");
        self.operator_l.as_ref().map(|loc| loc.print("operator"));
        self.expression_l.print("expression");
    }
}

