use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalList;
#[cfg(feature = "compile-with-external-structures")]
type List<T> = ExternalList<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type List<T> = Vec<T>;


#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalMaybeLoc;
#[cfg(feature = "compile-with-external-structures")]
type MaybeLoc = ExternalMaybeLoc;
#[cfg(not(feature = "compile-with-external-structures"))]
type MaybeLoc = Option<Loc>;


/// Represents compound statement (i.e. a multi-statement)
///
/// Basically all blocks of code are wrapped into `Begin` node (e.g. method/block body, rescue/ensure handler etc)
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Begin {
    /// A list of statements
    ///
    pub statements: List<Node>,
    /// Begin of the block
    ///
    /// ```text
    /// (1; 2)
    /// ~
    /// ```
    ///
    /// `None` if the block of code is "implicit", like
    ///
    /// ```text
    /// if true; 1; 2; end
    /// ```
    ///
    pub begin_l: MaybeLoc,
    /// End of the block
    ///
    /// ```text
    /// (1; 2)
    ///      ~
    /// ```
    ///
    /// `None` if the block of code is "implicit", like
    ///
    /// ```text
    /// if true; 1; 2; end
    /// ```
    ///
    pub end_l: MaybeLoc,
    /// Location of the full expression
    ///
    /// ```text
    /// (1; 2)
    /// ~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Begin {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_nodes(&self.statements);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "begin"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        for node in self.statements.iter() {
            node.inner_ref().print_with_locs();
        }
        self.begin_l.as_ref().map(|loc| loc.print("begin"));
        self.end_l.as_ref().map(|loc| loc.print("end"));
        self.expression_l.print("expression");
    }
}

