use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalStringPtr;
#[cfg(feature = "compile-with-external-structures")]
type StringPtr = ExternalStringPtr;
#[cfg(not(feature = "compile-with-external-structures"))]
type StringPtr = String;


/// Represents a positional required block/method argument.
///
/// `a` in `def m(a); end` or `proc { |a| }`
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Arg {
    /// Name of the argument
    ///
    pub name: StringPtr,
    /// Location of the full expression
    ///
    /// ```text
    /// def m(argument); end
    ///       ~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Arg {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_str(&self.name);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "arg"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.expression_l.print("expression");
    }
}

