use crate::nodes::InnerNode;
use crate::nodes::InspectVec;
use crate::Loc;
use crate::Node;

#[cfg(feature = "compile-with-external-structures")]
use crate::containers::ExternalPtr;
#[cfg(feature = "compile-with-external-structures")]
type Ptr<T> = ExternalPtr<T>;
#[cfg(not(feature = "compile-with-external-structures"))]
type Ptr<T> = Box<T>;


/// Represents `alias to from` statement.
///
#[derive(Debug, Clone, PartialEq)]
#[repr(C)]
pub struct Alias {
    /// Target of the `alias`.
    ///
    /// `Sym("foo")` node for `alias :foo :bar`
    ///
    pub to: Ptr<Node>,
    /// Source of the `alias`.
    ///
    /// `Sym("bar")` node for `alias :foo :bar`
    ///
    pub from: Ptr<Node>,
    /// Location of the `alias` keyword
    ///
    /// ```text
    /// alias foo bar
    /// ~~~~~
    /// ```
    ///
    pub keyword_l: Loc,
    /// Location of the full expression
    ///
    /// ```text
    /// alias foo bar
    /// ~~~~~~~~~~~~~
    /// ```
    ///
    pub expression_l: Loc,
}

impl InnerNode for Alias {
    fn expression(&self) -> &Loc {
        &self.expression_l
    }

    fn inspected_children(&self, indent: usize) -> Vec<String> {
        let mut result = InspectVec::new(indent);
        result.push_node(&self.to);
        result.push_node(&self.from);
        result.strings()
    }

    fn str_type(&self) -> &'static str {
        "alias"
    }

    fn print_with_locs(&self) {
        println!("{}", self.inspect(0));
        self.to.inner_ref().print_with_locs();
        self.from.inner_ref().print_with_locs();
        self.keyword_l.print("keyword");
        self.expression_l.print("expression");
    }
}

