use crate::{helpers, Options, NS as GLOBAL_NS};
use lib_ruby_parser_nodes::{nodes, Node};

pub fn node(options: &Options) -> String {
    format!(
        "
// Node

// Node constructors
{constructors}

// Node variant predicates
{variant_predicates}

// Node variant getter
{variant_getters}

// Node field getters
{field_getters}

// Node field setters
{field_setters}

#ifdef __cplusplus
extern \"C\" {{
#endif

{internal_structs}

#ifdef __cplusplus
}}
#endif

// into_variant fns
{into_variant_fns}

// into_internal fns
{into_internal_fns}

// variant drop fns
{variant_drop_fns}

void {fn_attributes} {ns}__node__drop({node_blob}* self_blob);
",
        ns = GLOBAL_NS,
        fn_attributes = options.fn_attributes,
        constructors = constructors(&options),
        variant_predicates = variant_predicates(&options),
        variant_getters = variant_getters(&options),
        field_getters = field_getters(&options),
        field_setters = field_setters(&options),
        internal_structs = internal_structs(&options),
        into_variant_fns = into_variant_fns(&options),
        into_internal_fns = into_internal_fns(&options),
        variant_drop_fns = variant_drop_fns(&options),
        node_blob = options.node_blob_name
    )
}

fn constructors(options: &Options) -> String {
    map_nodes_to_sigs(options, &helpers::nodes::constructor::sig)
}
fn variant_predicates(options: &Options) -> String {
    map_nodes_to_sigs(options, &helpers::nodes::variant_predicate::sig)
}
fn variant_getters(options: &Options) -> String {
    map_nodes_to_sigs(options, &helpers::nodes::variant_getter::sig)
}
fn field_getters(options: &Options) -> String {
    nodes()
        .flat_map(&|node| {
            node.fields.map(&|field| {
                format!(
                    "{sig};",
                    sig = helpers::nodes::field_getter::sig(node, field, options)
                )
            })
        })
        .join("\n")
}
fn field_setters(options: &Options) -> String {
    nodes()
        .flat_map(&|node| {
            node.fields.map(&|field| {
                format!(
                    "{sig};",
                    sig = helpers::nodes::field_setter::sig(node, field, options)
                )
            })
        })
        .join("\n")
}
fn internal_structs(options: &Options) -> String {
    nodes()
        .map(&|node| helpers::nodes::internal_struct::definition(node, options))
        .join("\n")
}
fn into_variant_fns(options: &Options) -> String {
    map_nodes_to_sigs(options, &helpers::nodes::into_variant::sig)
}
fn into_internal_fns(options: &Options) -> String {
    map_nodes_to_sigs(options, &helpers::nodes::into_internal::sig)
}
fn variant_drop_fns(options: &Options) -> String {
    map_nodes_to_sigs(options, &helpers::nodes::drop_variant::sig)
}

fn map_nodes_to_sigs(options: &Options, sig_fn: &dyn Fn(&Node, &Options) -> String) -> String {
    nodes()
        .map(&|node| format!("{sig};", sig = sig_fn(node, options)))
        .join("\n")
}
