use lib_ruby_parser_nodes::{Message, Node};

/// A struct with codegen options
pub struct Options {
    /// Identifier that is passed to wrapping `#ifndef`
    ///
    /// Default: `LIB_RUBY_PARSER_BINDINGS_H`
    pub ifndef_name: String,

    /// Pre-code
    ///
    /// You can put here things like `#include` or
    /// `#ifdef __cplusplus \n extern "C" ...`
    ///
    /// Default: `"// pre-code"`
    pub pre_code: String,

    /// Post-code
    ///
    /// Can be used to close `#ifdef __cplusplus`
    ///
    /// Default: `"// post-code"`
    pub post_code: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Ptr<T>` struct in `lib-ruby-parser`
    pub ptr_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybePtr<T>` struct in `lib-ruby-parser`
    pub maybe_ptr_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `StringPtr` struct in `lib-ruby-parser`
    pub string_ptr_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeStringPtr` struct in `lib-ruby-parser`
    pub maybe_string_ptr_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `SharedByteList` struct in `lib-ruby-parser`
    pub shared_byte_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `u8` struct in `lib-ruby-parser`
    ///
    /// In most cases if you build low-level bindings it will be u8
    /// (without converting it to/from blob)
    pub byte_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Token` struct in `lib-ruby-parser`
    pub token_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Node` struct in `lib-ruby-parser`
    pub node_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Diagnostic` struct in `lib-ruby-parser`
    pub diagnostic_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `CommentType` struct in `lib-ruby-parser`
    pub comment_type_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Comment` struct in `lib-ruby-parser`
    pub comment_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MagicCommentKind` struct in `lib-ruby-parser`
    pub magic_comment_kind_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MagicComment` struct in `lib-ruby-parser`
    pub magic_comment_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Bytes` struct in `lib-ruby-parser`
    pub bytes_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Loc` struct in `lib-ruby-parser`
    pub loc_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeLoc` struct in `lib-ruby-parser`
    pub maybe_loc_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `ErrorLevel` struct in `lib-ruby-parser`
    pub error_level_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Diagnostic` struct in `lib-ruby-parser`
    pub diagnostic_message_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `SourceLine` struct in `lib-ruby-parser`
    pub source_line_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Byte>` struct in `lib-ruby-parser`
    pub byte_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Token>` struct in `lib-ruby-parser`
    pub token_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Node>` struct in `lib-ruby-parser`
    pub node_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Diagnostic>` struct in `lib-ruby-parser`
    pub diagnostic_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Comment>` struct in `lib-ruby-parser`
    pub comment_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<MagicComment>` struct in `lib-ruby-parser`
    pub magic_comment_list_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `List<SourceLine>` struct in `lib-ruby-parser`
    pub source_line_list_blob_name: String,

    /// Function that takes a `Message` from `lib-ruby-parser-nodes`
    /// and returns a name of what you want to be used
    /// **as a blob** of its representation
    /// in in `lib-ruby-parser`
    ///
    /// For example, for `UnexpectedToken` message you may want to
    /// call your blob class/struct in C++ as `UnexpectedToken_BLOB`
    ///
    /// Default: fn that returns `<message.camelcase_name>_BLOB`
    pub message_variant_blob_name_fn: Box<dyn Fn(&Message) -> String>,

    /// Function that takes a `Node` from `lib-ruby-parser-nodes`
    /// and returns a name of what you want to be used
    /// **as a blob** of its representation
    /// in in `lib-ruby-parser`
    ///
    /// For example, for `Arg` message you may want to
    /// call your blob class/struct in C++ as `Arg_BLOB`
    ///
    /// Default: fn that returns `<node.camelcase_name>_BLOB`
    pub node_variant_blob_name_fn: Box<dyn Fn(&Node) -> String>,

    /// Name of what you want to be used
    /// **as a blob** of the `InputError` struct in `lib-ruby-parser`
    pub input_error_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `DecoderResult` struct in `lib-ruby-parser`
    pub decoder_result_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `Decoder` struct in `lib-ruby-parser`
    pub decoder_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `RewriteAction` struct in `lib-ruby-parser`
    pub rewrite_action_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `LexStateAction` struct in `lib-ruby-parser`
    pub lex_state_action_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `TokenRewriterResult` struct in `lib-ruby-parser`
    pub token_rewriter_result_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `TokenRewriter` struct in `lib-ruby-parser`
    pub token_rewriter_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeDecoder` struct in `lib-ruby-parser`
    pub maybe_decoder_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeTokenRewriter` struct in `lib-ruby-parser`
    pub maybe_token_rewriter_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `ParserOptions` struct in `lib-ruby-parser`
    pub parser_options_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `DecodedInput` struct in `lib-ruby-parser`
    pub decoded_input_blob_name: String,

    /// Name of what you want to be used
    /// **as a blob** of the `ParserResult` struct in `lib-ruby-parser`
    pub parser_result_blob_name: String,
}

impl Default for Options {
    fn default() -> Self {
        Self {
            ifndef_name: String::from("LIB_RUBY_PARSER_BINDINGS_H"),
            pre_code: String::from("// pre-code"),
            post_code: String::from("// post-code"),

            // blob names
            ptr_blob_name: String::from("Ptr_BLOB"),
            maybe_ptr_blob_name: String::from("MaybePtr_BLOB"),
            string_ptr_blob_name: String::from("StringPtr_BLOB"),
            maybe_string_ptr_blob_name: String::from("MaybeStringPtr_BLOB"),
            shared_byte_list_blob_name: String::from("SharedByteList_BLOB"),
            byte_blob_name: String::from("Byte_BLOB"),
            token_blob_name: String::from("Token_BLOB"),
            node_blob_name: String::from("Node_BLOB"),
            diagnostic_blob_name: String::from("Diagnostic_BLOB"),
            comment_type_blob_name: String::from("CommentType_BLOB"),
            comment_blob_name: String::from("Comment_BLOB"),
            magic_comment_kind_blob_name: String::from("MagicCommentKind_BLOB"),
            magic_comment_blob_name: String::from("MagicComment_BLOB"),
            source_line_blob_name: String::from("SourceLine_BLOB"),
            bytes_blob_name: String::from("Bytes_BLOB"),
            loc_blob_name: String::from("Loc_BLOB"),
            maybe_loc_blob_name: String::from("MaybeLoc_BLOB"),
            error_level_blob_name: String::from("ErrorLevel_BLOB"),
            diagnostic_message_blob_name: String::from("DiagnosticMessage_BLOB"),

            // list blob names
            byte_list_blob_name: String::from("ByteList_BLOB"),
            token_list_blob_name: String::from("TokenList_BLOB"),
            node_list_blob_name: String::from("NodeList_BLOB"),
            diagnostic_list_blob_name: String::from("DiagnosticList_BLOB"),
            comment_list_blob_name: String::from("CommentList_BLOB"),
            magic_comment_list_blob_name: String::from("MagicCommentList_BLOB"),
            source_line_list_blob_name: String::from("SourceLineList_BLOB"),

            // messages
            message_variant_blob_name_fn: Box::new(&|message: &Message| {
                format!("{}_BLOB", message.camelcase_name())
            }),

            // nodes
            node_variant_blob_name_fn: Box::new(&|node: &Node| {
                format!("{}_BLOB", node.camelcase_name)
            }),

            input_error_blob_name: String::from("InputError_BLOB"),
            decoder_result_blob_name: String::from("DecoderResult_BLOB"),
            decoder_blob_name: String::from("Decoder_BLOB"),

            rewrite_action_blob_name: String::from("RewriteAction_BLOB"),
            lex_state_action_blob_name: String::from("LexStateAction_BLOB"),
            token_rewriter_result_blob_name: String::from("TokenRewriterResult_BLOB"),
            token_rewriter_blob_name: String::from("TokenRewriter_BLOB"),

            maybe_decoder_blob_name: String::from("MaybeDecoder_BLOB"),
            maybe_token_rewriter_blob_name: String::from("MaybeTokenRewriter_BLOB"),
            parser_options_blob_name: String::from("ParserOptions_BLOB"),

            decoded_input_blob_name: String::from("DecodedInput_BLOB"),
            parser_result_blob_name: String::from("ParserResult_BLOB"),
        }
    }
}

impl Options {
    /// Applies `self.message_variant_blob_name_fn` on a given `message`
    ///
    /// Utility function
    pub fn message_variant_blob_name(&self, message: &Message) -> String {
        (&*self.message_variant_blob_name_fn)(message)
    }

    /// Applies `self.node_variant_blob_name` on a given `node`
    ///
    /// Utility function
    pub fn node_variant_blob_name(&self, node: &Node) -> String {
        (&*self.node_variant_blob_name_fn)(node)
    }
}
